/******************************************************************************
Copyright (c) 2009-2013 TP-Link Technologies CO.,LTD.  All rights reserved. 

File name	: br_deliver.c
Version		: v1.0, release version
Description	: This file impleaments the dual wifi repeater connection func
		  for some dual band repeaters, such as wa3500re(EU) 1.0.
 
Author		: huanglifu <huanglifu@tp-link.net>
Create date	: 2013/5/27

History		:
01, 2013/05/27 huanglifu, Created file.
02, 2013/09/23 tengfei, Modify file name and some descriptions to remove info
               related special product(wda3150).
03, 2014/07/11 Zhou Guofeng ,add the proc file for naming the devive ,debug  and 
              fix the bug that block in-loop packet, add non-connection filter table 
03, 2014/10/08 Zhou Guofeng ,remove 2g_cli_enable and 5g_cli_enable, using link_state instead.        
*****************************************************************************/
#include "br_deliver.h"
#include <linux/proc_fs.h>
#include <net/arp.h>

/*========================================================*/
/*================            GLOBAL VARIABLES             ================*/
/*========================================================*/

#ifndef ETHER_TYPE_802_1X
#define	ETHER_TYPE_802_1X	0x888e		/* 802.1x */
#endif


/*global variable for free bridge fdb.*/
struct tp_db_fre_bridge_fdb_entry free_bridage_fdb;

/*create a proc dir for user layer controling.*/
struct proc_dir_entry * range_extender_config_entry = NULL;
EXPORT_SYMBOL(range_extender_config_entry);

/*a switch for enable range_extender bridge deliver function.*/
int g_range_extender_bridge_deliver_enable = 0;
static struct proc_dir_entry * range_extender_bridge_deliver_enable_entry = NULL;


/*set the link status of front ,
	LINK_STATE_NONE = 0,
	LINK_STATE_ONLY_2G = 2,
	LINK_STATE_ONLY_5G = 4,
	LINK_STATE_ALL_2G_5G =6
*/
int g_range_extender_link_state =0 ;
static struct proc_dir_entry * range_extender_link_state_entry=NULL;

/*a flag for dev eth/br0 using 2g or 5g link to front */
//change by zgf , default value is setting to 0 ,eth lead to 5G.
int g_range_extender_eth_to_2g_enable = 0;

static struct proc_dir_entry *range_extender_eth_to_2g_enable_entry = NULL;

/*when range_extender wireless useing 3addrees, set this variable value as 1.*/
int g_range_extender_wireless_repeater_3addrees = 1;
static struct proc_dir_entry *range_extender_wireless_repeater_3addrees_entry = NULL;

/*when range_extender dual wifi connect to same front dut, set this variable value as 1.*/
int g_range_extender_has_same_front_dut = 1;
static struct proc_dir_entry *range_extender_has_same_front_dut_entry = NULL;

/*port name define file*/
char g_name_matrix[BRIDGE_DELIVER_DEV_NUM][BRIDGE_DELIVER_DEV_NAME_LEN]={"\0"};

static struct proc_dir_entry *range_extender_br_dev_name=NULL;

/* use for debug when it is setting to 1*/
int g_range_extender_dbg=0;
static struct proc_dir_entry *range_extender_dbg=NULL;

/*show the filter table*/
static struct proc_dir_entry *range_extender_filter_table=NULL;

/*the  index of the proxy sta address alias rules */
int g_range_extender_psr_alias_rule=0;
static struct proc_dir_entry *range_extender_psr_alias_rule=NULL;

/*========================================================*/
/*================            FUNCTION             ======================*/
/*========================================================*/

static void init_filter_table(struct tp_db_fre_bridge_fdb_entry *p)
{
	if (NULL == p)
	{	
		return;
	}
	
	memset(&p->filter_table_2g_5g, 0, sizeof(p->filter_table_2g_5g));
	memset(&p->filter_table_only_2g, 0, sizeof(p->filter_table_only_2g));
	memset(&p->filter_table_only_5g, 0, sizeof(p->filter_table_only_5g));
	memset(&p->filter_table_none, 0, sizeof(p->filter_table_none));
	
/*
 * FILETER RULE TABLE FOR 2G & 5G.
 * when value is 1, it mean that from dev should not deliver to dev. else is allowed.
 * from\to| AP2G | AP5G | CLI2G | CLI5G | BR0 | ETH 
 * AP2G   |  -      |    0    |     0    |   1      |  0    |  0
 * AP5G   |  0      |  -      |    1     |   0      |  0    |  0
 * CLI2G  |  0      |  1      |    -     |   1      |  0    |  0
 * CLI5G  |  1      |  0      |    1     |   -      |  1    |  1
 * BR0     |  0      |  0      |    0     |   1      |  -    |  0
 * ETH     |  0      |  0      |    0     |   1      |  0    |  -
*/
	FILTER_SET_VALUE_ON(p->filter_table_2g_5g, p->client_dev_2g_idx, p->client_dev_5g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_2g_5g, p->client_dev_2g_idx, p->ap_dev_5g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_2g_5g, p->ap_dev_5g_idx, p->client_dev_2g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_2g_5g, p->ap_dev_2g_idx, p->client_dev_5g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_2g_5g, p->client_dev_5g_idx, p->client_dev_2g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_2g_5g, p->client_dev_5g_idx, p->ap_dev_2g_idx);	

	FILTER_SET_VALUE_ON(p->filter_table_2g_5g, p->br0_dev_idx, 
		GET_2G_5G_DEV_INDEX_BY_FLAG(p->eth_goto_2g_flag, p->client_dev_5g_idx, p->client_dev_2g_idx));	
	FILTER_SET_VALUE_ON(p->filter_table_2g_5g, 
		GET_2G_5G_DEV_INDEX_BY_FLAG(p->eth_goto_2g_flag, p->client_dev_5g_idx, p->client_dev_2g_idx), 
		p->br0_dev_idx);	

	FILTER_SET_VALUE_ON(p->filter_table_2g_5g, p->eth_dev_idx, 
		GET_2G_5G_DEV_INDEX_BY_FLAG(p->eth_goto_2g_flag, p->client_dev_5g_idx, p->client_dev_2g_idx));
	FILTER_SET_VALUE_ON(p->filter_table_2g_5g, 
		GET_2G_5G_DEV_INDEX_BY_FLAG(p->eth_goto_2g_flag, p->client_dev_5g_idx, p->client_dev_2g_idx), 
		p->eth_dev_idx);

/*
 * FILETER TABLE FOR 2G ONLY.
 * when value is 1, it mean that from dev should not deliver to dev. else is allowed.
 * from\to| AP2G | AP5G | CLI2G | CLI5G | BR0 | ETH 
 * AP2G   |  -   |  0   |   0   |   1   |  0  |  0
 * AP5G   |  0   |  -   |   0   |   1   |  0  |  0
 * CLI2G  |  0   |  0   |   -   |   1   |  0  |  0
 * CLI5G  |  1   |  1   |   1   |   -   |  1  |  1
 * BR0    |  0   |  0   |   0   |   1   |  -  |  0
 * ETH    |  0   |  0   |   0   |   1   |  0  |  -
*/
	FILTER_SET_VALUE_ON(p->filter_table_only_2g, p->client_dev_5g_idx, p->client_dev_2g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_only_2g, p->client_dev_5g_idx, p->ap_dev_2g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_only_2g, p->client_dev_5g_idx, p->ap_dev_5g_idx);	
	FILTER_SET_VALUE_ON(p->filter_table_only_2g, p->client_dev_5g_idx, p->br0_dev_idx);
	FILTER_SET_VALUE_ON(p->filter_table_only_2g, p->client_dev_5g_idx, p->eth_dev_idx);
	
	FILTER_SET_VALUE_ON(p->filter_table_only_2g, p->ap_dev_5g_idx, p->client_dev_5g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_only_2g, p->client_dev_2g_idx, p->client_dev_5g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_only_2g, p->ap_dev_2g_idx, p->client_dev_5g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_only_2g, p->eth_dev_idx, p->client_dev_5g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_only_2g, p->br0_dev_idx, p->client_dev_5g_idx);	
/*
 * FILETER TABLE FOR 5G ONLY.
 * when value is 1, it mean that from dev should not deliver to dev. else is allowed.
 * from\to| AP2G | AP5G | CLI2G | CLI5G | BR0 | ETH 
 * AP2G   |  -   |  0   |   1   |   0   |  0  |  0
 * AP5G   |  0   |  -   |   1   |   0   |  0  |  0
 * CLI2G  |  1   |  1   |   -   |   1   |  1  |  1
 * CLI5G  |  0   |  0   |   1   |   -   |  0  |  0
 * BR0    |  0   |  0   |   1   |   0   |  -  |  0
 * ETH    |  0   |  0   |   1   |   0   |  0  |  -
*/

	FILTER_SET_VALUE_ON(p->filter_table_only_5g, p->client_dev_2g_idx, p->client_dev_5g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_only_5g, p->client_dev_2g_idx, p->ap_dev_2g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_only_5g, p->client_dev_2g_idx, p->ap_dev_5g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_only_5g, p->client_dev_2g_idx, p->br0_dev_idx);
	FILTER_SET_VALUE_ON(p->filter_table_only_5g, p->client_dev_2g_idx, p->eth_dev_idx);
	
	FILTER_SET_VALUE_ON(p->filter_table_only_5g, p->ap_dev_5g_idx, p->client_dev_2g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_only_5g, p->client_dev_5g_idx, p->client_dev_2g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_only_5g, p->ap_dev_2g_idx, p->client_dev_2g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_only_5g, p->eth_dev_idx, p->client_dev_2g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_only_5g, p->br0_dev_idx, p->client_dev_2g_idx);
/*
 * FILETER TABLE FOR NONE Connection
 * when value is 1, it mean that from dev should not deliver to dev. else is allowed.
 * from\to| AP2G | AP5G | CLI2G | CLI5G | BR0 | ETH 
 * AP2G   |    -    |    0    |    1     |   1      |    0   |  0
 * AP5G   |    0    |    -    |    1     |   1      |    0   |  0
 * CLI2G  |    1    |    1    |    -     |   1      |    1   |  1
 * CLI5G  |    1    |    1    |    1     |   -      |    0   |  1
 * BR0     |     0   |    0    |    1     |   0      |     -  |  0
 * ETH     |     0   |    0    |    1     |   1      |     0   |  -
*/
	FILTER_SET_VALUE_ON(p->filter_table_none, p->ap_dev_2g_idx, p->client_dev_2g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_none, p->ap_dev_2g_idx, p->client_dev_5g_idx);
	
	FILTER_SET_VALUE_ON(p->filter_table_none, p->ap_dev_5g_idx, p->client_dev_2g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_none, p->ap_dev_5g_idx, p->client_dev_5g_idx)

	FILTER_SET_VALUE_ON(p->filter_table_none, p->client_dev_2g_idx, p->ap_dev_2g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_none, p->client_dev_2g_idx, p->ap_dev_5g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_none, p->client_dev_2g_idx, p->client_dev_5g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_none, p->client_dev_2g_idx, p->eth_dev_idx);
	
	FILTER_SET_VALUE_ON(p->filter_table_none, p->client_dev_5g_idx, p->ap_dev_2g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_none, p->client_dev_5g_idx, p->ap_dev_5g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_none, p->client_dev_5g_idx, p->client_dev_2g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_none, p->client_dev_5g_idx, p->eth_dev_idx);

	FILTER_SET_VALUE_ON(p->filter_table_none, p->br0_dev_idx, 
		GET_2G_5G_DEV_INDEX_BY_FLAG(p->eth_goto_2g_flag, p->client_dev_5g_idx, p->client_dev_2g_idx));	
	FILTER_SET_VALUE_ON(p->filter_table_none, 
		GET_2G_5G_DEV_INDEX_BY_FLAG(p->eth_goto_2g_flag, p->client_dev_5g_idx, p->client_dev_2g_idx), 
		p->br0_dev_idx);	

	FILTER_SET_VALUE_ON(p->filter_table_none, p->eth_dev_idx, p->client_dev_2g_idx);
	FILTER_SET_VALUE_ON(p->filter_table_none, p->eth_dev_idx, p->client_dev_5g_idx);
	
	return;
}


static void br_do_send_arp_request_packet_proxy(void);

/*========================================================*/
/*================           PROC FILE W/R FUNCTION             ============*/
/*========================================================*/

static int range_extender_bridge_deliver_enable_entry_read (char *page, char **start, off_t off,
                               int count, int *eof, void *data)
{
	GET_PROC_VAR_VALUE(g_range_extender_bridge_deliver_enable);
}

static int range_extender_bridge_deliver_enable_entry_write (struct file *file, const char *buf,
                                        unsigned long count, void *data)
{
	SET_PROC_VAR_VALUE(g_range_extender_bridge_deliver_enable);

	REPEATER_PRINT(("%s %d range_extender_bridge_deliver_enable = %d\n"
		,__FUNCTION__, __LINE__, g_range_extender_bridge_deliver_enable));
	free_bridage_fdb.is_enabled = 0;
	return count;
}


static int range_extender_link_state_entry_read(char *page, char **start, off_t off,
                               int count, int *eof, void *data)
{
	 GET_PROC_VAR_VALUE(g_range_extender_link_state);

}

static int range_extender_link_state_entry_write (struct file *file, const char *buf,
                                        unsigned long count, void *data)
{
	int old_link_state;

	u_int32_t val;

	if (sscanf(buf, "%d", &val) != 1)
	        return -EINVAL;
	if ((val < 0) || (val > 6))
	{
		return -EINVAL;
	}

	g_range_extender_link_state= val;
	
	old_link_state = free_bridage_fdb.link_state;
	
	free_bridage_fdb.link_state = g_range_extender_link_state;
	
	REPEATER_PRINT(("free_bridage_fdb.link_state = %d\n", free_bridage_fdb.link_state));

	if(g_range_extender_wireless_repeater_3addrees
		&& free_bridage_fdb.is_enabled 
		&& old_link_state != free_bridage_fdb.link_state)
	{
		br_do_send_arp_request_packet_proxy();
	}	
	return count;
}

static int range_extender_eth_to_2g_enable_entry_read (char *page, char **start, off_t off,
                               int count, int *eof, void *data)
{
    GET_PROC_VAR_VALUE(g_range_extender_eth_to_2g_enable);
}

static int range_extender_eth_to_2g_enable_entry_write (struct file *file, const char *buf,
                                        unsigned long count, void *data)
{
	int old_eth_goto_2g_flag;
	SET_PROC_VAR_VALUE(g_range_extender_eth_to_2g_enable);
	old_eth_goto_2g_flag = free_bridage_fdb.eth_goto_2g_flag;
	REPEATER_PRINT(("range_extender_eth_to_2g_enable = %d\n", g_range_extender_eth_to_2g_enable));
	free_bridage_fdb.eth_goto_2g_flag = g_range_extender_eth_to_2g_enable;

	if(g_range_extender_wireless_repeater_3addrees 
		&& free_bridage_fdb.is_enabled 
		&& old_eth_goto_2g_flag != free_bridage_fdb.eth_goto_2g_flag)
	{
		init_filter_table(&free_bridage_fdb);
		br_do_send_arp_request_packet_proxy();
	}
	
	return count;
}

static int range_extender_wireless_repeater_3addrees_entry_read (char *page, char **start, off_t off,
                               int count, int *eof, void *data)
{
    GET_PROC_VAR_VALUE(g_range_extender_wireless_repeater_3addrees);
}

static int range_extender_wireless_repeater_3addrees_entry_write (struct file *file, const char *buf,
                                        unsigned long count, void *data)
{
	SET_PROC_VAR_VALUE(g_range_extender_wireless_repeater_3addrees);
	REPEATER_PRINT(("range_extender_wireless_repeater_3addrees = %d\n", g_range_extender_wireless_repeater_3addrees ));
	return count;
}

static int range_extender_has_same_front_dut_entry_read (char *page, char **start, off_t off,
                               int count, int *eof, void *data)
{
	g_range_extender_has_same_front_dut = free_bridage_fdb.is_same_front_dut;
    	GET_PROC_VAR_VALUE(g_range_extender_has_same_front_dut);
}

static int range_extender_has_same_front_dut_entry_write (struct file *file, const char *buf,
                                        unsigned long count, void *data)
{
	SET_PROC_VAR_VALUE(g_range_extender_has_same_front_dut);
	free_bridage_fdb.is_same_front_dut = g_range_extender_has_same_front_dut;
	REPEATER_PRINT(("g_range_extender_has_same_front_dut = %d\n", g_range_extender_has_same_front_dut ));
	return count;
}

static int range_extender_dbg_read (char *page, char **start, off_t off,
                               int count, int *eof, void *data)
{
    	GET_PROC_VAR_VALUE(g_range_extender_dbg);
}

static int range_extender_dbg_write (struct file *file, const char *buf,
                                        unsigned long count, void *data)
{
	SET_PROC_VAR_VALUE(g_range_extender_dbg);
	REPEATER_PRINT(("g_range_extender_dbg = %d\n", g_range_extender_dbg ));
	return count;
}

static int range_extender_filter_table_read (char *page, char **start, off_t off,
                               int count, int *eof, void *data)
{
	int i,j,ret,len=0;
	char *pos=page;

	if(off>0)
	{
		*eof=1;
		return 0;
	}
//filter_table_2g_5g	
	ret=sprintf(pos,"==================filter_table_2g_5g====================\n");
	len+=ret;
	pos+=ret;

	ret=sprintf(pos," \t0\t1\t2\t3\t4\t5\t6\n");
	len+=ret;
	pos+=ret;
	
	for(i=0;i<7;i++)
	{
		ret=sprintf(pos,"%d",i);
		len+=ret;
		pos+=ret;
		for(j=0;j<7;j++)
		{
			ret=sprintf(pos,"\t%d",free_bridage_fdb.filter_table_2g_5g[i][j]);
			len+=ret;
			pos+=ret;
		}
		ret=sprintf(pos,"\n");
		len+=ret;
		pos+=ret;
	}
	return len;
	
}


static int range_extender_br_dev_name_read (char *page, char **start, off_t off,
                               int count, int *eof, void *data)
{
	int i,ret,len=0;
    	char *pos=page;

	if(off>0)
	{
		*eof=1;
		return 0;
	}

	ret=sprintf(pos,"==================name define results====================\n");
	len+=ret;
	pos+=ret;

	for(i=0;i<BRIDGE_DELIVER_DEV_NUM;i++)
	{
		ret=sprintf(pos,"%s",g_name_matrix[i]);
		len+=ret;
		pos+=ret;

		ret=sprintf(pos,"\n");
		len+=ret;
		pos+=ret;
	}

	return len;
	
}

static int range_extender_br_dev_name_write (struct file *file, const char *buf,
                                        unsigned long count, void *data)
{	
	char name[BRIDGE_DELIVER_DEV_MAX_SIZE]={0};
	unsigned long max_length=count;
	int i=0,devindex=0,len=0;

	printk("1,count is %d\n",count);

	if(max_length>BRIDGE_DELIVER_DEV_MAX_SIZE)
		max_length=BRIDGE_DELIVER_DEV_MAX_SIZE;

	memset(name,0,BRIDGE_DELIVER_DEV_MAX_SIZE);

	if (copy_from_user(name,buf,max_length))
		return -EFAULT;

	
	while(i<=count)
	{

		printk("%c",name[i]);
		if(name[i]=='\0'||name[i]=='\n')
		{
			g_name_matrix[devindex++][len]='\0';
			break;
		}
		else if(name[i]==':')
		{
			g_name_matrix[devindex++][len]='\0';
			i++;
			len=0;
		}
		else
		{			
			g_name_matrix[devindex][len++]=name[i];
			i++;
		}
	}
	return count;
}


static int range_extender_psr_alias_rule_entry_read (char *page, char **start, off_t off,
                               int count, int *eof, void *data)
{
    GET_PROC_VAR_VALUE(g_range_extender_psr_alias_rule);
}

static int range_extender_psr_alias_rule_entry_write (struct file *file, const char *buf,
                                        unsigned long count, void *data)
{

	
	SET_PROC_VAR_VALUE(g_range_extender_psr_alias_rule);
		
	return count;

}

/*============================================================*/

int create_range_extender_config_proc(void)
{
	if (range_extender_config_entry != NULL)
	{
		printk ("Already have a proc entry for /proc/range_extender_config!\n");
		return -ENOENT;
	}

	range_extender_config_entry = proc_mkdir("range_extender_config", NULL);
	if (!range_extender_config_entry)
	{
		return -ENOENT;
	}

	range_extender_bridge_deliver_enable_entry = create_proc_entry("range_extender_bridge_deliver_enable", 0666, range_extender_config_entry);
	if (!range_extender_bridge_deliver_enable_entry)
	{
		return -ENOENT;
	}
	range_extender_bridge_deliver_enable_entry->read_proc = range_extender_bridge_deliver_enable_entry_read;
	range_extender_bridge_deliver_enable_entry->write_proc = range_extender_bridge_deliver_enable_entry_write; 


	range_extender_link_state_entry= create_proc_entry("range_extender_link_state", 0666, range_extender_config_entry);
	if (!range_extender_link_state_entry)
	{
		return -ENOENT;
	}
	range_extender_link_state_entry->read_proc = range_extender_link_state_entry_read;
	range_extender_link_state_entry->write_proc = range_extender_link_state_entry_write; 

	
	range_extender_eth_to_2g_enable_entry = create_proc_entry("range_extender_eth_to_2g_enable", 0666, range_extender_config_entry);
	if (!range_extender_eth_to_2g_enable_entry)
	{
		return -ENOENT;
	}

	range_extender_eth_to_2g_enable_entry->read_proc = range_extender_eth_to_2g_enable_entry_read;
	range_extender_eth_to_2g_enable_entry->write_proc = range_extender_eth_to_2g_enable_entry_write; 

	range_extender_wireless_repeater_3addrees_entry = create_proc_entry("range_extender_repeater_is_3addrees", 0666, range_extender_config_entry);
	if (!range_extender_wireless_repeater_3addrees_entry)
	{
		return -ENOENT;
	}
	range_extender_wireless_repeater_3addrees_entry->read_proc = range_extender_wireless_repeater_3addrees_entry_read;
	range_extender_wireless_repeater_3addrees_entry->write_proc = range_extender_wireless_repeater_3addrees_entry_write; 

	range_extender_has_same_front_dut_entry = create_proc_entry("range_extender_front_dut_num", 0666, range_extender_config_entry);
	if (!range_extender_has_same_front_dut_entry)
	{
		return -ENOENT;
	}
	range_extender_has_same_front_dut_entry->read_proc = range_extender_has_same_front_dut_entry_read;
	range_extender_has_same_front_dut_entry->write_proc = range_extender_has_same_front_dut_entry_write; 

	/* For dbg */
	range_extender_dbg = create_proc_entry("range_extender_dbg", 0666, range_extender_config_entry);
	if (!range_extender_dbg)
	{
		return -ENOENT;
	}
	range_extender_dbg->read_proc = range_extender_dbg_read;
	range_extender_dbg->write_proc = range_extender_dbg_write; 

	//For print the Filter tables

	range_extender_filter_table=create_proc_entry("range_extender_filter_table", 0666, range_extender_config_entry);
	if (!range_extender_filter_table)
	{
		return -ENOENT;
	}
	range_extender_filter_table->read_proc = range_extender_filter_table_read;
	range_extender_filter_table->write_proc = NULL; 

	// for the br dev name defined
	range_extender_br_dev_name=create_proc_entry("range_extender_br_dev_name", 0666, range_extender_config_entry);
	if (!range_extender_br_dev_name)
	{
		printk ("Can't Creat  Entry range_extender_br_dev_name !\n");
		return -ENOENT;
	}
	range_extender_br_dev_name->read_proc = range_extender_br_dev_name_read;
	range_extender_br_dev_name->write_proc = range_extender_br_dev_name_write; 

	// for the setting of the alias rules
	range_extender_psr_alias_rule=create_proc_entry("range_extender_psr_alias_rule",0666, range_extender_config_entry);
	if(!range_extender_psr_alias_rule)
	{
		printk ("Can't Creat  Entry range_extender_psr_alias_rule!\n");
		return -ENOENT;
	}
	range_extender_psr_alias_rule->read_proc=range_extender_psr_alias_rule_entry_read;
	range_extender_psr_alias_rule->write_proc=range_extender_psr_alias_rule_entry_write;

	return 0;

}

/*========================================================*/
/*================           BRIDGE DELIVER FILTER FUNCTION       =========*/
/*========================================================*/


int init_free_bridge_fdb(void)
{

	struct 	net_device* tmpDev = NULL;
	struct tp_db_fre_bridge_fdb_entry *p = &free_bridage_fdb;
	struct net_bridge_port* tmp_dev_br_port = NULL;
	unsigned int br_sub_dev_index_count = 0;
	
	if(LINK_STATE_ON != g_range_extender_bridge_deliver_enable)
	{
		goto INIT_FAIL;
	}

	if(NULL == p)
	{
		printk("bridge ERROR: NULL fdb\n");
		goto INIT_FAIL;
	}
	
	memset(p, 0, sizeof(struct tp_db_fre_bridge_fdb_entry));	
	
	br_sub_dev_index_count = 0;
	/*back of dual repeater*/
	GET_DEV_INDEX_BY_NAME(tmpDev, p->ap_dev_2g_idx, BRIDGE_DELIVER_2G_DEV_NAME);
	GET_DEV_INDEX_BY_NAME(tmpDev, p->ap_dev_5g_idx, BRIDGE_DELIVER_5G_DEV_NAME);	
	GET_DEV_INDEX_BY_NAME(tmpDev, p->eth_dev_idx, BRIDGE_DELIVER_LAN_ETH_NAME);	
	GET_DEV_INDEX_BY_NAME(tmpDev, p->br0_dev_idx, BRIDGE_DELIVER_NAME);	
	/*front of dual repeater*/
	GET_DEV_INDEX_BY_NAME(tmpDev, p->client_dev_2g_idx, BRIDGE_DELIVER_2G_STA_NAME);
	GET_DEV_INDEX_BY_NAME(tmpDev, p->client_dev_5g_idx, BRIDGE_DELIVER_5G_STA_NAME);
	
	GET_DEV_BY_NAME(p->client_2g_dev, BRIDGE_DELIVER_2G_STA_NAME);
	GET_DEV_BY_NAME(p->client_5g_dev, BRIDGE_DELIVER_5G_STA_NAME);	
	GET_DEV_BY_NAME(p->br0_dev, BRIDGE_DELIVER_NAME);	
	GET_DEV_BY_NAME(p->eth_dev, BRIDGE_DELIVER_LAN_ETH_NAME);	
	
	tmp_dev_br_port = br_port_get_rcu(p->client_2g_dev);
	if (tmp_dev_br_port == NULL)
	{
		goto INIT_FAIL;
	}
	p->br0_bri_dev = tmp_dev_br_port->br;
	
	p->is_enabled = LINK_STATE_ON;
	p->link_state = g_range_extender_link_state;
	p->eth_goto_2g_flag = g_range_extender_eth_to_2g_enable;	
	/* set its value as default value. */
	p->is_same_front_dut = g_range_extender_has_same_front_dut;

	/*
	**	Put init_filter_table at last to fix the problem that traffic from eth will go through apclii0 link
	@@	By TengFei.
	*/
	init_filter_table(&free_bridage_fdb);
	
	return LINK_STATE_ON;
	
INIT_FAIL:	
	p->is_enabled = LINK_STATE_OFF;
	g_range_extender_bridge_deliver_enable = 0;
	return LINK_STATE_OFF;
}

static unsigned int tp_get_br_front_port_ip(unsigned int src_ip)
{
	struct tp_db_fre_bridge_fdb_entry *p = &free_bridage_fdb;
 	struct net_bridge *br = p->br0_bri_dev;
	unsigned int dst_ip = 0;	
	int i;

	
	for (i = 0; i < BR_HASH_SIZE && !dst_ip; i++) {
		struct net_bridge_fdb_entry *f;
		struct hlist_node *h, *n;

		hlist_for_each_entry_safe(f, h, n, &br->hash[i], hlist) {			
		 	
			if (f->dst->dev->br_port_index == p->client_dev_2g_idx
				|| f->dst->dev->br_port_index == p->client_dev_5g_idx)
			{				
				if(src_ip != f->ip){
					dst_ip = f->ip;
					break;
				}
			}
		}
	}
	
	
	return dst_ip;	
}

static void br_do_br0_send_arp_request_packet_proxy(void)
{
	struct tp_db_fre_bridge_fdb_entry *p = &free_bridage_fdb;
	struct net_bridge *br = p->br0_bri_dev;
	struct net_device* tmp_dev = NULL;
	struct in_device *pdev_ipaddr=NULL;
	unsigned int dst_ip = 0;
	unsigned int src_ip = 0;
	//printk("++++++zgf%s %d \n", __FUNCTION__, __LINE__);
	if (!p->br0_dev || !p->br0_dev->ip_ptr)
	{
		return;
	}
	pdev_ipaddr = (struct in_device *)p->br0_dev->ip_ptr;
	if(!pdev_ipaddr || !pdev_ipaddr->ifa_list)
	{
		return;
	}
	src_ip = pdev_ipaddr->ifa_list->ifa_local;
	
	if (!src_ip)
	{
		return;
	}

	switch (p->link_state)
	{
		case LINK_STATE_ALL_2G_5G:
			tmp_dev = (p->eth_goto_2g_flag) ? p->client_2g_dev : p->client_5g_dev;
		break;
		
		case LINK_STATE_ONLY_2G:
			tmp_dev = p->client_2g_dev;
		break;
		
		case LINK_STATE_ONLY_5G:
			tmp_dev = p->client_5g_dev;
		break;	
	}

	/*
	**	Sometimes tmp_dev(client_2g_dev or client_5g_dev) would be free, and this may cause kernel panic.
	**	So it is necessary to check existence of "tmp_dev".
	@@	TengFei, 13/11/11
	*/
	
	if(!tmp_dev)
		return;
	
	spin_lock_bh(&br->hash_lock);
	dst_ip = tp_get_br_front_port_ip(src_ip);
	spin_unlock_bh(&br->hash_lock);	

	if (dst_ip)
	{ 
		arp_send(ARPOP_REQUEST, ETH_P_ARP, dst_ip, tmp_dev, src_ip, NULL, p->br0_dev->dev_addr, NULL);
	}

}

static void br_do_send_arp_request_packet_proxy(void)
{
	struct tp_db_fre_bridge_fdb_entry *p = &free_bridage_fdb;
 	struct net_bridge *br = p->br0_bri_dev;
	struct net_device* tmp_dev = NULL;
	unsigned int dst_ip = 0;
	int i;


	br_do_br0_send_arp_request_packet_proxy();
	
	spin_lock_bh(&br->hash_lock);
	
	for (i = 0; i < BR_HASH_SIZE; i++) {
		struct net_bridge_fdb_entry *f;
		struct hlist_node *h, *n;

		hlist_for_each_entry_safe(f, h, n, &br->hash[i], hlist) {
			if (f->dst->dev->br_port_index == p->client_dev_2g_idx
				|| f->dst->dev->br_port_index == p->client_dev_5g_idx
				|| !f->ip)
			{
				continue;
			}
			/*initial tmp_dev each loop*/
			tmp_dev=NULL;
			switch (p->link_state)
			{
				case LINK_STATE_ALL_2G_5G:
					if (f->dst->dev->br_port_index == p->eth_dev_idx
						|| f->dst->dev->br_port_index == p->br0_dev_idx)
					{
						tmp_dev = (p->eth_goto_2g_flag) ? p->client_2g_dev : p->client_5g_dev;
					}else if (f->dst->dev->br_port_index == p->ap_dev_2g_idx)
					{
						tmp_dev = p->client_2g_dev;
					}else if (f->dst->dev->br_port_index == p->ap_dev_5g_idx)
					{
						tmp_dev = p->client_5g_dev;
					}
					break;
				
				case LINK_STATE_ONLY_2G:
					tmp_dev = p->client_2g_dev;
					break;
				
				case LINK_STATE_ONLY_5G:
					tmp_dev = p->client_5g_dev;
					break;	
				case LINK_STATE_NONE:
					tmp_dev=NULL;
					break;
			}

			/*
			**	Sometimes tmp_dev(client_2g_dev or client_5g_dev) would be free, and this may cause kernel panic.
			**	So it is necerrary to check existence of "tmp_dev".
			@@	TengFei, 13/11/11
			*/
			if(!tmp_dev)
				continue;
			
			dst_ip = 0;
			dst_ip = tp_get_br_front_port_ip(f->ip);
	
			if (dst_ip&&((f->addr.addr[0]&2)==0))
			{
				arp_send(ARPOP_REQUEST,ETH_P_ARP, dst_ip, tmp_dev, f->ip, NULL, f->addr.addr, NULL);
			}
			
		}
	}
	printk("++++++zgf%s %d \n", __FUNCTION__, __LINE__);
	spin_unlock_bh(&br->hash_lock);
}


int br_should_deliver_packet(struct sk_buff *skb)
{
	//const unsigned char *mac_src = NULL;
	struct tp_db_fre_bridge_fdb_entry *p = &free_bridage_fdb;
	//int is_cli_2g_5g_mac = 0;
	int ret=1;
	bool is_local=0;
	CHECK_AND_INIT_TABLE(1);

	/*
		only check the psta mapped address ,as the back-end client mac address will be translated to the mapped addrress.

		which mac[0]&2=1;
	*/


	is_local=(eth_hdr(skb)->h_source[0]&2);

	if (is_local&&(skb->dev->br_port_index == p->client_dev_5g_idx
			|| skb->dev->br_port_index == p->client_dev_2g_idx))
	{
		ret=br_fdb_check_ring_packet(skb, p);
	}

	REPEATER_PRINT(("++++zgf :%s!!,source mac is %x:%x:%x:%x:%x:%x,proto is 0x%x,dev index is %d,local %d\n",
		ret ? "DELIVER":"DROP",
		eth_hdr(skb)->h_source[0],
		eth_hdr(skb)->h_source[1],
		eth_hdr(skb)->h_source[2],
		eth_hdr(skb)->h_source[3],
		eth_hdr(skb)->h_source[4],
		eth_hdr(skb)->h_source[5],
		eth_hdr(skb)->h_proto,
		skb->dev->br_port_index,
		is_local));
	REPEATER_PRINT(("++++zgf :%s!!,dest mac is %x:%x:%x:%x:%x:%x,proto is  0x%x,dev index is %d\n",
		ret ? "DELIVER":"DROP",
		eth_hdr(skb)->h_dest[0],
		eth_hdr(skb)->h_dest[1],
		eth_hdr(skb)->h_dest[2],
		eth_hdr(skb)->h_dest[3],
		eth_hdr(skb)->h_dest[4],
		eth_hdr(skb)->h_dest[5],
		eth_hdr(skb)->h_proto,
		skb->dev->br_port_index));
	return ret;

}

static int _cmp_mac(unsigned char *mac1, unsigned char *mac2)
{
	return ((mac1[0]!=mac2[0])||(mac1[1]!=mac2[1])||(mac1[2]!=mac2[2])||
				mac1[3]!=mac2[3]||(mac1[4]!=mac2[4])||(mac1[5]=mac2[5]));

}

int br_db_fre_should_deliver(struct net_bridge_port *prev, struct sk_buff *skb)
{	
	struct net_device *psrc_dev = skb->dev;
	struct net_device *pdst_dev = prev->dev;
	struct tp_db_fre_bridge_fdb_entry *p = &free_bridage_fdb;
	int result = 0;	

	if (psrc_dev == NULL || pdst_dev == NULL)
	{
		return 0;
	}
	
	CHECK_AND_INIT_TABLE(0);

	REPEATER_PRINT(("++++zgf :source mac is %x:%x:%x,from index %d to index %d\n",
		eth_hdr(skb)->h_source[3],
		eth_hdr(skb)->h_source[4],
		eth_hdr(skb)->h_source[5],
		psrc_dev->br_port_index,
		pdst_dev->br_port_index));

	if (DUAL_REPEATER_HAS_SAME_FROMT_DUT != p->is_same_front_dut)
	{
		return 0;
	}
	switch (p->link_state)
	{
		case LINK_STATE_ALL_2G_5G:
			if (IS_DEV_FILTERED(p->filter_table_2g_5g, psrc_dev, pdst_dev))
			{
				result = 1;
			}
			break;
		
		case LINK_STATE_ONLY_2G:
			if (IS_DEV_FILTERED(p->filter_table_only_2g, psrc_dev, pdst_dev))
			{
				result = 1;
			}
			break;
		
		case LINK_STATE_ONLY_5G:
			if (IS_DEV_FILTERED(p->filter_table_only_5g, psrc_dev, pdst_dev))
			{
				result = 1;
			}
			break;
		case LINK_STATE_NONE:
			if (IS_DEV_FILTERED(p->filter_table_none, psrc_dev, pdst_dev))
			{
				result=1;
			}
			break;
	}
	


	/*  if br0 send 802_1X packet to rootap. pass this packet .
	*    add by zgf.
	*/

	if(result==1&&psrc_dev->br_port_index==p->br0_dev_idx)
	{
		if(eth_hdr(skb)->h_proto==htons(ETHER_TYPE_802_1X))
		{
			result=0;
		}
	}

	REPEATER_PRINT(("++++zgf  after check:source mac is %x:%x:%x,result is %s,link_state is %d\n",
		eth_hdr(skb)->h_source[3],
		eth_hdr(skb)->h_source[4],
		eth_hdr(skb)->h_source[5],
		result ? "drop":"deliver",
		p->link_state));

	return result;
}

int br_db_fre_fdb_lookup(struct net_bridge_port **to, struct sk_buff *skb)
{	

	struct net_device *psrc_dev = skb->dev;
	struct net_device *pdst_dev = (*to)->dev;
	struct tp_db_fre_bridge_fdb_entry *p = &free_bridage_fdb;
	struct net_device* tmp_dev = NULL;
	struct net_bridge_port* tmp_dev_br_port = NULL;

	if (psrc_dev == NULL || pdst_dev == NULL)
	{
		return 0;
	}


	REPEATER_PRINT(("++++zgf :source mac is %x:%x:%x,from index %d to index %d\n",
		eth_hdr(skb)->h_source[3],
		eth_hdr(skb)->h_source[4],
		eth_hdr(skb)->h_source[5],
		psrc_dev->br_port_index,
		pdst_dev->br_port_index));


	/*  if br0 send 802_1X packet to rootap. pass this packet .
	*    add by zgf.
	*/
	
	if(psrc_dev->br_port_index==p->br0_dev_idx)
	{
		if(eth_hdr(skb)->h_proto==htons(ETHER_TYPE_802_1X))
		{
			return 0;
		}
	}


	if (DUAL_REPEATER_HAS_SAME_FROMT_DUT != p->is_same_front_dut)
	{
		return 0;
	}	

	tmp_dev = NULL;
	switch(p->link_state)
	{
		case LINK_STATE_ALL_2G_5G:						
			if (IS_DEV_FILTERED(p->filter_table_2g_5g, psrc_dev, pdst_dev)
				&&(pdst_dev->br_port_index == p->client_dev_2g_idx 
				|| pdst_dev->br_port_index == p->client_dev_5g_idx))
			{
				tmp_dev = (pdst_dev->br_port_index == p->client_dev_2g_idx) ? p->client_5g_dev : p->client_2g_dev;				
			}

			if(psrc_dev->br_port_index==p->br0_dev_idx)
			{
				tmp_dev_br_port= br_port_get_rcu(p->eth_dev);
				if(tmp_dev_br_port)
				{
					br_fdb_update_mapped_hwaddr(tmp_dev_br_port->br, tmp_dev_br_port, skb);
				}
			}	
		break;

		case LINK_STATE_ONLY_2G:
			 if (pdst_dev->br_port_index == p->client_dev_5g_idx)
			 {
			 	if(psrc_dev->br_port_index!=p->br0_dev_idx)
			 		tmp_dev = p->client_2g_dev;
			 }
			break;

		case LINK_STATE_ONLY_5G:
			if (pdst_dev->br_port_index == p->client_dev_2g_idx)
			 {
			 	if(psrc_dev->br_port_index!=p->br0_dev_idx)
			 		tmp_dev = p->client_5g_dev;
			 }
			break;
		case LINK_STATE_NONE:
			if(IS_DEV_FILTERED(p->filter_table_none, psrc_dev, pdst_dev)
				&&(pdst_dev->br_port_index == p->client_dev_2g_idx 
				|| pdst_dev->br_port_index == p->client_dev_5g_idx))
			{
				tmp_dev=psrc_dev;
			}			
				
	}

	if (tmp_dev && (tmp_dev_br_port = br_port_get_rcu(tmp_dev)))
	{
		*to = tmp_dev_br_port;
	}


	REPEATER_PRINT(("++++zgf :after check source mac is %x:%x:%x,from index %d to index %d,link state is %d\n",
		eth_hdr(skb)->h_source[3],
		eth_hdr(skb)->h_source[4],
		eth_hdr(skb)->h_source[5],
		psrc_dev->br_port_index,
		(*to)->dev->br_port_index,
		p->link_state));

	return 0;
}


/*
**	For MTK platform, we should update mapped mac by apcli0/apclii0 for ra0/rai0/eth2.1.
@@	TengFei, 14/04/10.
*/
unsigned br_fdb_update_mapped_hwaddr(struct net_bridge *br, struct net_bridge_port *source,
		   struct sk_buff *skb)
{
	struct tp_db_fre_bridge_fdb_entry *p = &free_bridage_fdb;
	struct net_device *tmp_dev = skb->dev;
	unsigned char mapped_mac[ETH_ALEN];
	unsigned char oui[ETH_ALEN/2];
	unsigned int m,b;

	CHECK_AND_INIT_TABLE(0);
	if(p->link_state==LINK_STATE_ALL_2G_5G)
	{
		if (tmp_dev)
		{	
			if (tmp_dev->br_port_index == p->ap_dev_2g_idx ||
				tmp_dev->br_port_index == p->ap_dev_5g_idx ||
				tmp_dev->br_port_index == p->eth_dev_idx||
				tmp_dev->br_port_index == p->br0_dev_idx)
			{
				memcpy(mapped_mac, eth_hdr(skb)->h_source, ETH_ALEN);
				
				if(g_range_extender_psr_alias_rule==1)
				{
					memcpy(oui,mapped_mac, ETH_ALEN/2);
					oui[0]=oui[0]&0xfd;

					if (memcmp(oui, p->client_5g_dev->dev_addr, ETH_ALEN/2)) {
						/* Use the oui from primary interface's hardware address */
						mapped_mac[0] = p->client_5g_dev->dev_addr[0];
						mapped_mac[1] = p->client_5g_dev->dev_addr[1];
						mapped_mac[2] = p->client_5g_dev->dev_addr[2];
		
						/* Set the locally administered bit */
						mapped_mac[0] |= 0x02;
					}
					else
					{
						/* Set the locally administered bit */
						mapped_mac[0] |= 0x02;
						/* Right rotate the octets[1:3] of the mac address. This will make
						 * sure we generate an unique fixed alias for each mac address. If two
	 					* client mac addresses have the same octets[1:3] then we will have
						 * a collision. If this happens then generate a random number for the
						 * mac address.
						 */
						m = mapped_mac[1] << 16 |mapped_mac[2] << 8 | mapped_mac[3];

						b = m & 1;
						m >>= 1;
						m |= (b << 23);

						mapped_mac[1] = m >> 16;
						mapped_mac[2] = (m >> 8) & 0xff;
						mapped_mac[3] = m & 0xff;
					}

				}			
				else
					mapped_mac[0]|=0x02;
				
				/* MAC Repeater has been improved */			
				br_fdb_update(br, source, mapped_mac);
			}
		}
	}

	return 0;
}

